<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Candidate;
use App\Models\Period;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;

class CandidateController extends Controller
{
    /**
     * Tampilkan daftar kandidat (per periode aktif) + pencarian.
     */
    public function index(Request $request)
    {
        $period = Period::where('is_active', 1)->latest()->first();
        $q = trim((string) $request->query('q', ''));

        $candidates = Candidate::when($period, fn ($qq) => $qq->where('period_id', $period->id))
            ->when($q !== '', function ($qq) use ($q) {
                $qq->where(function ($w) use ($q) {
                    $w->where('name', 'like', "%$q%")
                      ->orWhere('class', 'like', "%$q%");
                });
            })
            ->orderBy('sort_order')
            ->orderBy('name')
            ->paginate(12)
            ->withQueryString();

        return view('admin.candidates.index', compact('period', 'candidates', 'q'));
    }

    /**
     * Form tambah kandidat.
     */
    public function create()
    {
        $period = Period::where('is_active', 1)->latest()->first();
        return view('admin.candidates.create', compact('period'));
    }

    /**
     * Simpan kandidat baru.
     */
    public function store(Request $request)
    {
        $data = $request->validate([
            'name'       => 'required|string|max:100',
            'class'      => 'nullable|string|max:50',
            'vision'     => 'nullable|string',
            'mission'    => 'nullable|string',
            'sort_order' => 'nullable|integer|min:0|max:9999',
            'is_active'  => 'required|boolean',
            'photo'      => 'nullable|image|max:2048', // 2MB
        ]);

        $period = Period::where('is_active', 1)->latest()->first();
        $data['period_id']  = $period?->id;
        $data['sort_order'] = $data['sort_order'] ?? 0;

        if ($request->hasFile('photo')) {
            $data['photo_path'] = $this->storePhoto($request);
        }

        Candidate::create($data);

        return redirect()->route('admin.candidates.index')->with('status', 'Kandidat ditambahkan.');
    }

    /**
     * Form edit kandidat.
     */
    public function edit(Candidate $candidate)
    {
        return view('admin.candidates.edit', compact('candidate'));
    }

    /**
     * Update kandidat.
     */
    public function update(Request $request, Candidate $candidate)
    {
        $data = $request->validate([
            'name'       => 'required|string|max:100',
            'class'      => 'nullable|string|max:50',
            'vision'     => 'nullable|string',
            'mission'    => 'nullable|string',
            'sort_order' => 'nullable|integer|min:0|max:9999',
            'is_active'  => 'required|boolean',
            'photo'      => 'nullable|image|max:2048',
        ]);

        $data['sort_order'] = $data['sort_order'] ?? 0;

        if ($request->hasFile('photo')) {
            // hapus foto lama jika ada
            if ($candidate->photo_path && Storage::disk('public')->exists($candidate->photo_path)) {
                Storage::disk('public')->delete($candidate->photo_path);
            }
            $data['photo_path'] = $this->storePhoto($request);
        }

        $candidate->update($data);

        return redirect()->route('admin.candidates.index')->with('status', 'Kandidat diupdate.');
    }

    /**
     * Hapus kandidat (beserta fotonya).
     */
    public function destroy(Candidate $candidate)
    {
        if ($candidate->photo_path && Storage::disk('public')->exists($candidate->photo_path)) {
            Storage::disk('public')->delete($candidate->photo_path);
        }
        $candidate->delete();

        return back()->with('status', 'Kandidat dihapus.');
    }

    /**
     * Toggle aktif/nonaktif kandidat (AJAX atau form POST biasa).
     */
    public function toggleActive(Candidate $candidate)
    {
        $candidate->update(['is_active' => !$candidate->is_active]);
        return back()->with('status', 'Status kandidat diperbarui.');
    }

    /**
     * Reorder kandidat (drag & drop) — terima array [id => position] atau array urutan id.
     * Contoh payload:
     *  - positions: { "5": 0, "8": 1, "2": 2 }
     *  - order: [5,8,2]  (akan diset 0..n)
     */
    public function reorder(Request $request)
    {
        $request->validate([
            'positions' => 'array',
            'order'     => 'array',
        ]);

        DB::transaction(function () use ($request) {
            if ($request->filled('positions')) {
                foreach ($request->positions as $id => $pos) {
                    Candidate::whereKey($id)->update(['sort_order' => (int) $pos]);
                }
            } elseif ($request->filled('order')) {
                foreach ($request->order as $i => $id) {
                    Candidate::whereKey($id)->update(['sort_order' => (int) $i]);
                }
            }
        });

        return response()->json(['ok' => true]);
    }

    /**
     * Simpan foto ke storage/public/candidates dan mengembalikan path relatif.
     */
    private function storePhoto(Request $request): string
    {
        // pastikan sudah menjalankan: php artisan storage:link
        return $request->file('photo')->store('candidates', 'public');
    }
}
